<?php
// dashboard_api_fixed.php
require_once '../../config.php';

// connect
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    header('Content-Type: application/json');
    die(json_encode(['error' => 'Connection failed: ' . mysqli_connect_error()]));
}

// helper: binary IP → dotted quad
function binaryToIp($binaryIp) {
    if (strlen($binaryIp) === 4) {
        $u = unpack('N', $binaryIp);
        return long2ip($u[1]);
    }
    return null;
}

// FIXED: batch-geolocate IPs with proper chunking
function batchGetLocations($ips) {
    // Filter valid public IPs
    $valid = array_filter($ips, function($ip){
        return filter_var($ip, FILTER_VALIDATE_IP)
            && filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE);
    });
    
    if (!$valid) return [];
    
    // Remove duplicates to minimize API calls
    $valid = array_unique($valid);
    
    $allResults = [];
    $endpoint = 'http://ip-api.com/batch?fields=status,query,lat,lon,city,regionName';
    
    // Process in chunks of 100 (API limit)
    $chunks = array_chunk($valid, 100);
    
    foreach ($chunks as $chunkIndex => $chunk) {
        // Add small delay between chunks to avoid rate limiting
        if ($chunkIndex > 0) {
            usleep(100000); // 0.1 second delay
        }
        
        $opts = [
            'http' => [
                'method' => 'POST',
                'header' => 'Content-Type: application/json',
                'content' => json_encode(array_values($chunk)),
                'timeout' => 10 // Increased timeout for larger batches
            ]
        ];
        
        $ctx = stream_context_create($opts);
        $resp = @file_get_contents($endpoint, false, $ctx);
        
        if (!$resp) {
            // Log error but continue with other chunks
            error_log("IP batch lookup failed for chunk $chunkIndex");
            continue;
        }
        
        $decoded = json_decode($resp, true);
        if (!$decoded) continue;
        
        foreach ($decoded as $r) {
            if (!empty($r['status']) && $r['status'] === 'success') {
                $allResults[$r['query']] = [
                    'state' => $r['regionName'] ?? 'Unknown',
                    'city'  => $r['city'] ?? '',
                    'lat'   => $r['lat'] ?? null,
                    'lon'   => $r['lon'] ?? null,
                ];
            }
        }
    }
    
    return $allResults;
}

// avg online time
function calculateAverageOnlineTime($con) {
    $sql = "
      SELECT AVG(heartbeat * 10) AS avg_minutes
      FROM (
        SELECT mac_address, AVG(heartbeat) AS heartbeat
        FROM glance_scroll_tracking
        WHERE heartbeat > 1
        GROUP BY mac_address
      ) sub
    ";
    $res = mysqli_query($con, $sql);
    if (!$res) return 0;
    $row = mysqli_fetch_assoc($res);
    return round($row['avg_minutes'] ?? 0);
}

// grab latest session per MAC + map SN
$sql = "
  SELECT
    t1.mac_address,
    t1.ip_address,
    t1.session_count,
    t1.last_ping_time,
    sm.SN
  FROM glance_scroll_tracking t1
  JOIN (
    SELECT mac_address, MAX(session_count) AS max_session
    FROM glance_scroll_tracking
    GROUP BY mac_address
  ) t2 ON t1.mac_address = t2.mac_address
       AND t1.session_count = t2.max_session
  LEFT JOIN serial_mapping sm
    ON t1.mac_address = sm.MAC
  GROUP BY t1.mac_address
  ORDER BY t1.last_ping_time DESC
";
$result = mysqli_query($con, $sql);

$devices = [];
$onlineUsers = 0;
$stateDistribution = [];
$ipList = [];

if ($result) {
    // 1) collect IPs + raw rows
    $buffer = [];
    while ($r = mysqli_fetch_assoc($result)) {
        $ip = binaryToIp($r['ip_address']);
        if ($ip) {
            $ipList[] = $ip;
        }
        $buffer[] = ['row' => $r, 'ip' => $ip];
    }

    // 2) lookup geolocation with proper batching
    $locs = batchGetLocations($ipList);

    // 3) build devices array
    foreach ($buffer as $entry) {
        $r  = $entry['row'];
        $ip = $entry['ip'];

        // determine online/offline
        $q = "SELECT TIMESTAMPDIFF(MINUTE,'{$r['last_ping_time']}',NOW()) AS m";
        $t = mysqli_fetch_assoc(mysqli_query($con, $q))['m'];
        $isOnline = ($t <= 10);
        if ($isOnline) {
            $onlineUsers++;
        }

        // calculate current session online time
        $mac      = mysqli_real_escape_string($con, $r['mac_address']);
        $session  = (int)$r['session_count'];
        $hbSql    = "
            SELECT SUM(heartbeat) AS total_heartbeats
            FROM glance_scroll_tracking
            WHERE mac_address = '{$mac}'
              AND session_count = {$session}
        ";
        $hbRes    = mysqli_query($con, $hbSql);
        $hbRow    = mysqli_fetch_assoc($hbRes);
        $heartbeats = (int)($hbRow['total_heartbeats'] ?? 0);
        $minutes    = $heartbeats * 10;
        if ($minutes >= 60) {
            $hours = floor($minutes / 60);
            $mins  = $minutes % 60;
            $onlineTime = $hours . ' HR' . ($mins > 0 ? ' ' . $mins . ' min' : '');
        } else {
            $onlineTime = $minutes . ' min';
        }

        // geo info
        $state = 'Unknown';
        $city  = '';
        $lat   = null;
        $lon   = null;
        if ($ip && isset($locs[$ip])) {
            $state = $locs[$ip]['state'];
            $city  = $locs[$ip]['city'];
            $lat   = $locs[$ip]['lat'];
            $lon   = $locs[$ip]['lon'];
        } elseif ($ip && !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE)) {
            $state = 'Local Network';
        }
        
        // Always increment state distribution
        $stateDistribution[$state] = ($stateDistribution[$state] ?? 0) + 1;

        $devices[] = [
            'mac_address'         => $r['mac_address'],
            'sn'                  => $r['SN'] ?? null,
            'ip_address'          => $ip ?: 'Invalid IP',
            'state'               => $state,
            'city'                => $city,
            'lat'                 => $lat,
            'lon'                 => $lon,
            'session_count'       => $session,
            'last_seen'           => date('M j, g:i A', strtotime($r['last_ping_time'])),
            'isOnline'            => $isOnline,
            'current_online_time' => $onlineTime
        ];
    }
}

$avgOnlineMinutes = calculateAverageOnlineTime($con);
mysqli_close($con);

// Add debug info in development
$debugInfo = [];
if (isset($_GET['debug'])) {
    $debugInfo = [
        'total_ips_processed' => count($ipList),
        'unique_ips' => count(array_unique($ipList)),
        'locations_found' => count($locs),
        'chunks_processed' => ceil(count(array_unique($ipList)) / 100)
    ];
}

// output JSON
header('Content-Type: application/json');
$output = [
    'onlineUsers'          => $onlineUsers,
    'totalDevices'         => count($devices),
    'avgOnlineMinutes'     => $avgOnlineMinutes,
    'stateDistribution'    => $stateDistribution,
    'devices'              => $devices
];

if (!empty($debugInfo)) {
    $output['debug'] = $debugInfo;
}

echo json_encode($output);
?>