// GLANCE Mobile - Main JavaScript

// Global state
const AppState = {
    companyConfig: null,
    currentMAC: null,
    macMode: 'entered',
    selections: [],
    uploadCount: 0,
    uploadLimit: 100
};

// Initialize app
document.addEventListener('DOMContentLoaded', () => {u
    // Load company config if available
    const configData = document.getElementById('company-config');
    if (configData) {
        AppState.companyConfig = JSON.parse(configData.textContent);
        applyTheme(AppState.companyConfig.theme);
    }
    
    // Get current MAC if available (from PHP session)
    const macSelector = document.getElementById('mac-selector');
    if (macSelector && macSelector.value) {
        AppState.currentMAC = macSelector.value;
        AppState.macMode = 'precontrolled';
    }
    
    // Initialize components
    initializeLoginForm();
    initializeDeviceSelector();
    initializeSelectionGrid();
    initializeUploadModal();
    initializeMenu();
    initializeAddButton();
    
    // Update time display
    updateTimeDisplay();
    setInterval(updateTimeDisplay, 1000);
});

// Apply company theme
function applyTheme(theme) {
    const root = document.documentElement;
    Object.entries(theme).forEach(([key, value]) => {
        if (key.includes('color')) {
            root.style.setProperty(`--${key.replace('_', '-')}`, value);
        }
    });
}

// Update header time
function updateTimeDisplay() {
    const timeElement = document.querySelector('.header-time');
    if (timeElement) {
        const now = new Date();
        timeElement.textContent = now.toLocaleTimeString('en-US', {
            hour: 'numeric',
            minute: '2-digit',
            hour12: true
        });
    }
}

// Login form handler
function initializeLoginForm() {
    const loginForm = document.getElementById('login-form');
    if (!loginForm) return;
    
    loginForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const companyKey = document.getElementById('company-key').value.trim();
        const submitBtn = loginForm.querySelector('button[type="submit"]');
        
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="loading"></span> Verifying...';
        
        try {
            const response = await fetch('api/auth.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ company_key: companyKey })
            });
            
            const result = await response.json();
            
            if (result.success) {
                window.location.href = result.redirect || '?page=home';
            } else {
                showStatus(result.error || 'Invalid company key', 'error');
                submitBtn.disabled = false;
                submitBtn.textContent = 'Access Application';
            }
        } catch (error) {
            showStatus('Connection error. Please try again.', 'error');
            submitBtn.disabled = false;
            submitBtn.textContent = 'Access Application';
        }
    });
}

// Device selector
function initializeDeviceSelector() {
    // MAC input handler
    const macInput = document.getElementById('mac-input');
    const macSubmit = document.getElementById('mac-submit');
    
    if (macInput && macSubmit) {
        macSubmit.addEventListener('click', () => {
            const mac = macInput.value.trim().toUpperCase();
            if (validateMAC(mac)) {
                setDeviceMAC(mac, 'entered');
            } else {
                showStatus('Invalid MAC address format', 'error');
            }
        });
        
        macInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                macSubmit.click();
            }
        });
    }
    
    // Precontrolled MAC selector
    const macSelect = document.getElementById('mac-select');
    if (macSelect) {
        macSelect.addEventListener('change', () => {
            const mac = macSelect.value;
            if (mac) {
                setDeviceMAC(mac, 'precontrolled');
            }
        });
    }
}

// Validate MAC address format
function validateMAC(mac) {
    const macRegex = /^([0-9A-F]{2}[:-]){5}([0-9A-F]{2})$/i;
    return macRegex.test(mac);
}

// Set device MAC
async function setDeviceMAC(mac, mode) {
    try {
        const response = await fetch('api/device.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ mac, mode })
        });
        
        const result = await response.json();
        
        if (result.success) {
            AppState.currentMAC = mac;
            AppState.macMode = mode;
            window.location.href = '?page=home';
        } else {
            showStatus(result.error || 'Failed to set device', 'error');
        }
    } catch (error) {
        showStatus('Connection error', 'error');
    }
}

// Selection grid
function initializeSelectionGrid() {
    const grid = document.querySelector('.selection-grid');
    if (!grid) return;
    
    // Handle selection clicks
    grid.addEventListener('click', (e) => {
        const item = e.target.closest('.selection-item');
        if (!item) return;
        
        const contentId = item.dataset.contentId;
        if (!contentId) return;
        
        // Toggle selection
        if (item.classList.contains('selected')) {
            item.classList.remove('selected');
            AppState.selections = AppState.selections.filter(id => id !== contentId);
        } else {
            // Check selection limit (6 items max)
            if (AppState.selections.length >= 6) {
                showStatus('Maximum 6 selections allowed', 'warning');
                return;
            }
            
            item.classList.add('selected');
            AppState.selections.push(contentId);
        }
    });
}

// Initialize add button
function initializeAddButton() {
    const addBtn = document.querySelector('.add-button');
    if (!addBtn) return;
    
    addBtn.addEventListener('click', () => {
        showModal('add-content-modal');
    });
}

// Upload modal
function initializeUploadModal() {
    // Upload area drag & drop
    const uploadArea = document.querySelector('.upload-area');
    if (!uploadArea) return;
    
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        uploadArea.addEventListener(eventName, preventDefaults, false);
    });
    
    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    
    ['dragenter', 'dragover'].forEach(eventName => {
        uploadArea.addEventListener(eventName, () => {
            uploadArea.classList.add('dragover');
        });
    });
    
    ['dragleave', 'drop'].forEach(eventName => {
        uploadArea.addEventListener(eventName, () => {
            uploadArea.classList.remove('dragover');
        });
    });
    
    uploadArea.addEventListener('drop', handleDrop);
    
    // File input
    const fileInput = document.getElementById('image-upload');
    const uploadBtn = document.getElementById('upload-btn');
    
    if (uploadBtn && fileInput) {
        uploadBtn.addEventListener('click', () => fileInput.click());
        fileInput.addEventListener('change', handleFileSelect);
    }
    
    // Text content
    const addTextBtn = document.getElementById('add-text-btn');
    if (addTextBtn) {
        addTextBtn.addEventListener('click', handleAddText);
    }
}

// Handle file drop
function handleDrop(e) {
    const dt = e.dataTransfer;
    const files = dt.files;
    
    if (files.length > 0) {
        handleFile(files[0]);
    }
}

// Handle file selection
function handleFileSelect(e) {
    const files = e.target.files;
    if (files.length > 0) {
        handleFile(files[0]);
    }
}

// Handle single file
async function handleFile(file) {
    if (!file) return;
    
    // Check file type
    if (!file.type.startsWith('image/')) {
        showStatus('Please select an image file', 'error');
        return;
    }
    
    // Check file size (1MB limit)
    if (file.size > 1024 * 1024) {
        showStatus('File size must be less than 1MB', 'error');
        return;
    }
    
    // Show preview
    const reader = new FileReader();
    reader.onload = (e) => {
        const preview = document.getElementById('image-preview');
        if (preview) {
            preview.src = e.target.result;
            preview.style.display = 'block';
        }
    };
    reader.readAsDataURL(file);
    
    // Upload file
    const formData = new FormData();
    formData.append('image', file);
    formData.append('name', document.getElementById('image-name').value || file.name);
    
    try {
        const uploadBtn = document.getElementById('save-image-btn');
        uploadBtn.disabled = true;
        uploadBtn.innerHTML = '<span class="loading"></span> Uploading...';
        
        const response = await fetch('api/upload.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showStatus('Image uploaded successfully', 'success');
            closeModal('upload-modal');
            // Clear form
            document.getElementById('image-upload').value = '';
            document.getElementById('image-name').value = '';
            document.getElementById('image-preview').style.display = 'none';
            // Reload content library
            if (window.loadContentLibrary) {
                window.loadContentLibrary();
            }
        } else {
            showStatus(result.error || 'Upload failed', 'error');
        }
    } catch (error) {
        showStatus('Upload error: ' + error.message, 'error');
    } finally {
        const uploadBtn = document.getElementById('save-image-btn');
        uploadBtn.disabled = false;
        uploadBtn.textContent = 'Save Image';
    }
}

// Handle add text
async function handleAddText() {
    const textContent = document.getElementById('text-content').value.trim();
    const textColor = document.getElementById('text-color').value;
    
    if (!textContent) {
        showStatus('Please enter text content', 'error');
        return;
    }
    
    if (textContent.length > 255) {
        showStatus('Text must be less than 255 characters', 'error');
        return;
    }
    
    try {
        const saveBtn = document.getElementById('save-text-btn');
        saveBtn.disabled = true;
        saveBtn.innerHTML = '<span class="loading"></span> Saving...';
        
        const response = await fetch('api/add-text.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                text: textContent,
                color: textColor
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showStatus('Text added successfully', 'success');
            closeModal('text-modal');
            document.getElementById('text-content').value = '';
            // Reload content library
            if (window.loadContentLibrary) {
                window.loadContentLibrary();
            }
        } else {
            showStatus(result.error || 'Failed to add text', 'error');
        }
    } catch (error) {
        showStatus('Error: ' + error.message, 'error');
    } finally {
        const saveBtn = document.getElementById('save-text-btn');
        saveBtn.disabled = false;
        saveBtn.textContent = 'Save Text';
    }
}

// Menu handling
function initializeMenu() {
    const hamburger = document.querySelector('.hamburger');
    const sideMenu = document.querySelector('.side-menu');
    const menuOverlay = document.querySelector('.menu-overlay');
    
    if (!hamburger || !sideMenu || !menuOverlay) return;
    
    hamburger.addEventListener('click', () => {
        sideMenu.classList.add('active');
        menuOverlay.classList.add('active');
    });
    
    menuOverlay.addEventListener('click', () => {
        sideMenu.classList.remove('active');
        menuOverlay.classList.remove('active');
    });
    
    // Clear selections handlers
    const clearAllBtn = document.getElementById('clear-all-selections');
    const clearCompanyBtn = document.getElementById('clear-company-selections');
    
    if (clearAllBtn) {
        clearAllBtn.addEventListener('click', () => clearSelections(true));
    }
    
    if (clearCompanyBtn) {
        clearCompanyBtn.addEventListener('click', () => clearSelections(false));
    }
}

// Clear selections
async function clearSelections(clearAll) {
    const confirmMsg = clearAll ? 
        'Clear all selections for this device?' : 
        'Clear company selections for this device?';
        
    if (!confirm(confirmMsg)) return;
    
    try {
        const response = await fetch('api/clear-selections.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                mac: AppState.currentMAC,
                clear_all: clearAll
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showStatus('Selections cleared', 'success');
            // Reload page to refresh selections
            window.location.reload();
        } else {
            showStatus(result.error || 'Failed to clear selections', 'error');
        }
    } catch (error) {
        showStatus('Error: ' + error.message, 'error');
    }
}

// Save selections
async function saveSelections() {
    try {
        const saveBtn = document.getElementById('save-selections-btn');
        saveBtn.disabled = true;
        saveBtn.innerHTML = '<span class="loading"></span> Saving...';
        
        const response = await fetch('api/save-selections.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                mac: AppState.currentMAC,
                selections: AppState.selections
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showStatus('Selections saved successfully', 'success');
        } else {
            showStatus(result.error || 'Failed to save selections', 'error');
        }
    } catch (error) {
        showStatus('Error: ' + error.message, 'error');
    } finally {
        const saveBtn = document.getElementById('save-selections-btn');
        saveBtn.disabled = false;
        saveBtn.textContent = 'Save Selections';
    }
}

// Modal functions
function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('active');
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('active');
    }
}

// Status messages
function showStatus(message, type = 'info') {
    const statusContainer = document.getElementById('status-container') || createStatusContainer();
    
    const statusEl = document.createElement('div');
    statusEl.className = `status-message status-${type}`;
    statusEl.textContent = message;
    
    statusContainer.appendChild(statusEl);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        statusEl.remove();
    }, 5000);
}

function createStatusContainer() {
    const container = document.createElement('div');
    container.id = 'status-container';
    container.style.position = 'fixed';
    container.style.top = '70px';
    container.style.left = '50%';
    container.style.transform = 'translateX(-50%)';
    container.style.width = '90%';
    container.style.maxWidth = '400px';
    container.style.zIndex = '2000';
    document.body.appendChild(container);
    return container;
}

// Switch MAC address (for dropdown)
async function switchMAC(newMAC) {
    if (!newMAC) return;
    
    try {
        const response = await fetch('api/device.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ mac: newMAC, mode: 'precontrolled' })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Clear previous device data from session
            AppState.currentMAC = newMAC;
            AppState.macMode = 'precontrolled';
            // Reload to update the interface
            window.location.reload();
        } else {
            showStatus(result.error || 'Failed to switch device', 'error');
            // Reset dropdown to current value if failed
            const selector = document.getElementById('mac-selector');
            if (selector && AppState.currentMAC) {
                selector.value = AppState.currentMAC;
            }
        }
    } catch (error) {
        showStatus('Connection error', 'error');
        // Reset dropdown to current value if failed
        const selector = document.getElementById('mac-selector');
        if (selector && AppState.currentMAC) {
            selector.value = AppState.currentMAC;
        }
    }
}

// Attach to window for global access
window.switchMAC = switchMAC;
window.saveSelections = saveSelections;
window.showModal = showModal;
window.closeModal = closeModal;
window.handleFile = handleFile;
window.handleAddText = handleAddText;