<?php
/**
 * Device Management API Endpoint
 */

require_once '../../lib/database.php';
require_once '../../lib/functions.php';

header('Content-Type: application/json');

// Check authentication
if (!isAuthenticated('jackmorton')) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Handle POST requests only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['mac']) || !isset($input['mode'])) {
    echo json_encode(['success' => false, 'error' => 'MAC address and mode required']);
    exit;
}

$mac = strtoupper(sanitizeInput($input['mac']));
$mode = sanitizeInput($input['mode']);

// Validate MAC format
if (!preg_match('/^([0-9A-F]{2}[:-]){5}([0-9A-F]{2})$/i', $mac)) {
    echo json_encode(['success' => false, 'error' => 'Invalid MAC address format']);
    exit;
}

// Validate mode
if (!in_array($mode, ['entered', 'precontrolled'])) {
    echo json_encode(['success' => false, 'error' => 'Invalid mode']);
    exit;
}

// Load config
$config = getCompanyConfig('jackmorton');

// For precontrolled mode, validate MAC exists in database
if ($mode === 'precontrolled') {
    if (!validateMAC($mac, $config['comp_group'])) {
        echo json_encode(['success' => false, 'error' => 'MAC address not found in company devices']);
        exit;
    }
}

// Set the MAC address
setCurrentMAC($mac, $mode);

echo json_encode(['success' => true, 'mac' => $mac, 'mode' => $mode]);