<?php
/**
 * GLANCE Mobile - Image Processing Functions
 */

/**
 * Process uploaded image - resize to 192x32 and convert to binary
 * @param string $sourcePath Source image path
 * @param string $destinationPath Destination PNG path
 * @param string $binaryPath Binary output path
 * @return array ['success' => bool, 'error' => string|null]
 */
function processUploadedImage($sourcePath, $destinationPath, $binaryPath) {
    try {
        // Validate source file
        if (!file_exists($sourcePath)) {
            return ['success' => false, 'error' => 'Source file not found'];
        }
        
        // Get image info
        $imageInfo = getimagesize($sourcePath);
        if (!$imageInfo) {
            return ['success' => false, 'error' => 'Invalid image file'];
        }
        
        // Create image resource based on type
        $sourceImage = null;
        switch ($imageInfo['mime']) {
            case 'image/jpeg':
                $sourceImage = imagecreatefromjpeg($sourcePath);
                break;
            case 'image/png':
                $sourceImage = imagecreatefrompng($sourcePath);
                break;
            case 'image/gif':
                $sourceImage = imagecreatefromgif($sourcePath);
                break;
            case 'image/webp':
                $sourceImage = imagecreatefromwebp($sourcePath);
                break;
            default:
                return ['success' => false, 'error' => 'Unsupported image format'];
        }
        
        if (!$sourceImage) {
            return ['success' => false, 'error' => 'Failed to create image resource'];
        }
        
        // Create 192x32 canvas
        $targetImage = imagecreatetruecolor(192, 32);
        
        // Preserve transparency for PNG/GIF
        if ($imageInfo['mime'] === 'image/png' || $imageInfo['mime'] === 'image/gif') {
            imagealphablending($targetImage, false);
            imagesavealpha($targetImage, true);
            $transparent = imagecolorallocatealpha($targetImage, 255, 255, 255, 127);
            imagefilledrectangle($targetImage, 0, 0, 192, 32, $transparent);
        } else {
            // Fill with black background for JPEG
            $black = imagecolorallocate($targetImage, 0, 0, 0);
            imagefilledrectangle($targetImage, 0, 0, 192, 32, $black);
        }
        
        // Calculate scaling
        $sourceWidth = imagesx($sourceImage);
        $sourceHeight = imagesy($sourceImage);
        
        // Scale to fit within 192x32 maintaining aspect ratio
        $scaleX = 192 / $sourceWidth;
        $scaleY = 32 / $sourceHeight;
        $scale = min($scaleX, $scaleY);
        
        $newWidth = (int)($sourceWidth * $scale);
        $newHeight = (int)($sourceHeight * $scale);
        
        // Center the image
        $destX = (int)((192 - $newWidth) / 2);
        $destY = (int)((32 - $newHeight) / 2);
        
        // Copy and resize
        imagecopyresampled(
            $targetImage, $sourceImage,
            $destX, $destY, 0, 0,
            $newWidth, $newHeight,
            $sourceWidth, $sourceHeight
        );
        
        // Save as PNG
        $saveResult = imagepng($targetImage, $destinationPath, 9);
        
        if (!$saveResult) {
            imagedestroy($sourceImage);
            imagedestroy($targetImage);
            return ['success' => false, 'error' => 'Failed to save PNG image'];
        }
        
        // Convert to binary format for LED panel
        $binaryResult = convertImageToBinary($targetImage, $binaryPath);
        
        // Clean up
        imagedestroy($sourceImage);
        imagedestroy($targetImage);
        
        if (!$binaryResult['success']) {
            return $binaryResult;
        }
        
        return ['success' => true, 'error' => null];
        
    } catch (Exception $e) {
        error_log("Image processing error: " . $e->getMessage());
        return ['success' => false, 'error' => 'Image processing failed: ' . $e->getMessage()];
    }
}

/**
 * Convert image to binary format for LED panel
 * @param resource $image GD image resource
 * @param string $outputPath Binary file output path
 * @return array ['success' => bool, 'error' => string|null]
 */
function convertImageToBinary($image, $outputPath) {
    try {
        // Ensure output directory exists
        $outputDir = dirname($outputPath);
        if (!is_dir($outputDir)) {
            mkdir($outputDir, 0755, true);
        }
        
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Binary data array
        $binaryData = [];
        
        // Process each pixel
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $rgb = imagecolorat($image, $x, $y);
                $colors = imagecolorsforindex($image, $rgb);
                
                // Convert to RGB565 format (common for LED panels)
                // 5 bits red, 6 bits green, 5 bits blue
                $r = ($colors['red'] >> 3) & 0x1F;
                $g = ($colors['green'] >> 2) & 0x3F;
                $b = ($colors['blue'] >> 3) & 0x1F;
                
                $rgb565 = ($r << 11) | ($g << 5) | $b;
                
                // Store as 2 bytes (big-endian)
                $binaryData[] = ($rgb565 >> 8) & 0xFF;
                $binaryData[] = $rgb565 & 0xFF;
            }
        }
        
        // Write binary file
        $binaryString = '';
        foreach ($binaryData as $byte) {
            $binaryString .= chr($byte);
        }
        
        if (file_put_contents($outputPath, $binaryString) === false) {
            return ['success' => false, 'error' => 'Failed to write binary file'];
        }
        
        // Also create a JSON metadata file
        $metadataPath = str_replace('.bin', '_meta.json', $outputPath);
        $metadata = [
            'width' => $width,
            'height' => $height,
            'format' => 'RGB565',
            'created' => date('Y-m-d H:i:s'),
            'size' => strlen($binaryString)
        ];
        
        file_put_contents($metadataPath, json_encode($metadata, JSON_PRETTY_PRINT));
        
        return ['success' => true, 'error' => null];
        
    } catch (Exception $e) {
        error_log("Binary conversion error: " . $e->getMessage());
        return ['success' => false, 'error' => 'Binary conversion failed: ' . $e->getMessage()];
    }
}

/**
 * Create text image and convert to binary
 * @param string $text Text to render
 * @param string $color Hex color code
 * @param string $outputPath PNG output path
 * @param string $binaryPath Binary output path
 * @return array ['success' => bool, 'error' => string|null]
 */
function createTextImage($text, $color, $outputPath, $binaryPath) {
    try {
        // Create 192x32 canvas
        $image = imagecreatetruecolor(192, 32);
        
        // Set background color (black)
        $bgColor = imagecolorallocate($image, 0, 0, 0);
        imagefilledrectangle($image, 0, 0, 192, 32, $bgColor);
        
        // Parse hex color
        $hex = str_replace('#', '', $color);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $textColor = imagecolorallocate($image, $r, $g, $b);
        
        // TODO: Update font path to your system font
        $fontPath = '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf';
        
        // Try alternative font paths if primary doesn't exist
        if (!file_exists($fontPath)) {
            $alternativePaths = [
                '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf',
                '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',
                '/usr/share/fonts/liberation/LiberationSans-Regular.ttf',
                '/System/Library/Fonts/Helvetica.ttc',
                'C:\Windows\Fonts\arial.ttf'
            ];
            
            foreach ($alternativePaths as $altPath) {
                if (file_exists($altPath)) {
                    $fontPath = $altPath;
                    break;
                }
            }
        }
        
        // If no font found, use built-in font
        if (!file_exists($fontPath)) {
            // Use built-in font as fallback
            $font = 5; // Largest built-in font
            $textWidth = imagefontwidth($font) * strlen($text);
            $textHeight = imagefontheight($font);
            $x = (int)((192 - $textWidth) / 2);
            $y = (int)((32 - $textHeight) / 2);
            imagestring($image, $font, $x, $y, $text, $textColor);
        } else {
            // Calculate font size to fit text
            $fontSize = 20;
            $maxWidth = 180; // Leave some padding
            
            // Adjust font size to fit
            do {
                $bbox = imagettfbbox($fontSize, 0, $fontPath, $text);
                $textWidth = abs($bbox[4] - $bbox[0]);
                if ($textWidth <= $maxWidth) break;
                $fontSize--;
            } while ($fontSize > 8);
            
            // Center text
            $x = (int)((192 - $textWidth) / 2);
            $y = (int)(32 / 2 + $fontSize / 2);
            
            // Draw text
            imagettftext($image, $fontSize, 0, $x, $y, $textColor, $fontPath, $text);
        }
        
        // Save PNG
        if (!imagepng($image, $outputPath, 9)) {
            imagedestroy($image);
            return ['success' => false, 'error' => 'Failed to save text image'];
        }
        
        // Convert to binary
        $binaryResult = convertImageToBinary($image, $binaryPath);
        
        imagedestroy($image);
        
        return $binaryResult;
        
    } catch (Exception $e) {
        error_log("Text image creation error: " . $e->getMessage());
        return ['success' => false, 'error' => 'Failed to create text image: ' . $e->getMessage()];
    }
}

/**
 * Validate uploaded file
 * @param array $file $_FILES array element
 * @param array $config Company configuration
 * @return array ['valid' => bool, 'error' => string|null]
 */
function validateUploadedFile($file, $config) {
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['valid' => false, 'error' => 'Upload failed with error code: ' . $file['error']];
    }
    
    // Check file size
    $maxSize = ($config['features']['content_types']['image']['max_size_kb'] ?? 1024) * 1024;
    if ($file['size'] > $maxSize) {
        return ['valid' => false, 'error' => 'File size exceeds limit of ' . ($maxSize / 1024) . 'KB'];
    }
    
    // Check MIME type
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($mimeType, $allowedTypes)) {
        return ['valid' => false, 'error' => 'Invalid file type. Allowed types: JPEG, PNG, GIF, WebP'];
    }
    
    // Validate it's actually an image
    $imageInfo = getimagesize($file['tmp_name']);
    if (!$imageInfo) {
        return ['valid' => false, 'error' => 'File is not a valid image'];
    }
    
    return ['valid' => true, 'error' => null];
}