<?php
/**
 * Add Text Content API Endpoint
 */

require_once '../../lib/database.php';
require_once '../../lib/functions.php';
require_once '../../lib/image_processing.php';

header('Content-Type: application/json');

// Check authentication
if (!isAuthenticated('jackmorton')) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Handle POST requests only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['text']) || !isset($input['color'])) {
    echo json_encode(['success' => false, 'error' => 'Text and color required']);
    exit;
}

// Load config
$config = getCompanyConfig('jackmorton');

// Check upload limit (text counts towards daily limit)
$uploadStatus = checkUploadLimit($config['comp_group']);
if (!$uploadStatus['allowed']) {
    echo json_encode([
        'success' => false, 
        'error' => 'Daily content limit reached (' . $uploadStatus['count'] . '/' . $uploadStatus['limit'] . ')'
    ]);
    exit;
}

// Validate text
$text = sanitizeInput($input['text']);
$color = sanitizeInput($input['color']);

if (empty($text)) {
    echo json_encode(['success' => false, 'error' => 'Text cannot be empty']);
    exit;
}

if (strlen($text) > $config['features']['content_types']['text']['max_length']) {
    echo json_encode(['success' => false, 'error' => 'Text exceeds maximum length']);
    exit;
}

// Validate color format
if (!preg_match('/^#[0-9A-F]{6}$/i', $color)) {
    $color = $config['features']['content_types']['text']['default_color'];
}

// Create directories if they don't exist
$uploadBase = realpath(__DIR__ . '/../../../uploads') ?: __DIR__ . '/../../../uploads';
$companyUploadDir = $uploadBase . '/' . $config['comp_group'];
$binOutputDir = $uploadBase . '/' . $config['comp_group'] . '/bin';

if (!is_dir($companyUploadDir)) {
    mkdir($companyUploadDir, 0755, true);
}
if (!is_dir($binOutputDir)) {
    mkdir($binOutputDir, 0755, true);
}

// Generate text image
$textName = 'text_' . substr($text, 0, 20) . '_' . time();
$textName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $textName);
$uniqueName = generateUniqueFilename($textName, 'png', $companyUploadDir);
$pngPath = $companyUploadDir . '/' . $uniqueName;
$binPath = $binOutputDir . '/' . str_replace('.png', '.bin', $uniqueName);

// Create text image
$createResult = createTextImage($text, $color, $pngPath, $binPath);

if (!$createResult['success']) {
    echo json_encode(['success' => false, 'error' => $createResult['error']]);
    exit;
}

// Add to content library
$contentId = addContentToLibrary(
    $config['comp_group'],
    'text',
    substr($text, 0, 50) . (strlen($text) > 50 ? '...' : ''),
    [
        'text' => $text,
        'color' => $color,
        'created_date' => date('Y-m-d H:i:s')
    ],
    $pngPath
);

if (!$contentId) {
    // Clean up files if database insert fails
    @unlink($pngPath);
    @unlink($binPath);
    echo json_encode(['success' => false, 'error' => 'Failed to save to library']);
    exit;
}

// Increment upload count
incrementUploadCount($config['comp_group']);

echo json_encode([
    'success' => true,
    'content_id' => $contentId,
    'message' => 'Text content added successfully'
]);