<?php
/**
 * Image Upload API Endpoint
 */

require_once '../../lib/database.php';
require_once '../../lib/functions.php';
require_once '../../lib/image_processing.php';

header('Content-Type: application/json');

// Check authentication
if (!isAuthenticated('jackmorton')) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Handle POST requests only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Load config
$config = getCompanyConfig('jackmorton');

// Check upload limit
$uploadStatus = checkUploadLimit($config['comp_group']);
if (!$uploadStatus['allowed']) {
    echo json_encode([
        'success' => false, 
        'error' => 'Daily upload limit reached (' . $uploadStatus['count'] . '/' . $uploadStatus['limit'] . ')'
    ]);
    exit;
}

// Check if file was uploaded
if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
    echo json_encode(['success' => false, 'error' => 'No file uploaded or upload error']);
    exit;
}

// Validate uploaded file
$validation = validateUploadedFile($_FILES['image'], $config);
if (!$validation['valid']) {
    echo json_encode(['success' => false, 'error' => $validation['error']]);
    exit;
}

// Get image name
$imageName = sanitizeInput($_POST['name'] ?? pathinfo($_FILES['image']['name'], PATHINFO_FILENAME));
if (empty($imageName)) {
    $imageName = 'image_' . time();
}

// Create upload directories if they don't exist
$uploadBase = realpath(__DIR__ . '/../../../uploads') ?: __DIR__ . '/../../../uploads';
$companyUploadDir = $uploadBase . '/' . $config['comp_group'];
$binOutputDir = $uploadBase . '/' . $config['comp_group'] . '/bin';

// TODO: Update these paths based on your server structure
if (!is_dir($companyUploadDir)) {
    mkdir($companyUploadDir, 0755, true);
}
if (!is_dir($binOutputDir)) {
    mkdir($binOutputDir, 0755, true);
}

// Generate unique filename
$uniqueName = generateUniqueFilename($imageName, 'png', $companyUploadDir);
$pngPath = $companyUploadDir . '/' . $uniqueName;
$binPath = $binOutputDir . '/' . str_replace('.png', '.bin', $uniqueName);

// Process the uploaded image
$processResult = processUploadedImage($_FILES['image']['tmp_name'], $pngPath, $binPath);

if (!$processResult['success']) {
    echo json_encode(['success' => false, 'error' => $processResult['error']]);
    exit;
}

// Add to content library
$contentId = addContentToLibrary(
    $config['comp_group'],
    'image',
    $imageName,
    [
        'original_name' => $_FILES['image']['name'],
        'file_size' => $_FILES['image']['size'],
        'upload_date' => date('Y-m-d H:i:s')
    ],
    $pngPath
);

if (!$contentId) {
    // Clean up files if database insert fails
    @unlink($pngPath);
    @unlink($binPath);
    echo json_encode(['success' => false, 'error' => 'Failed to save to library']);
    exit;
}

// Increment upload count
incrementUploadCount($config['comp_group']);

echo json_encode([
    'success' => true,
    'content_id' => $contentId,
    'filename' => $uniqueName,
    'message' => 'Image uploaded successfully'
]);