<?php
/**
 * Home Page - Device Overview and Current Selections
 */

// Get device data
$deviceData = getDeviceSelections($currentMAC);
$selections = $deviceData['Selections'] ?? [];

// Get content library for display
$contentLibrary = getContentLibrary($config['comp_group']);
$contentMap = [];
foreach ($contentLibrary as $item) {
    $contentMap[$item['id']] = $item;
}

// Get MAC mode
$macMode = getMACMode();

// Get available MACs if in pre-controlled mode
$availableMACs = [];
if ($macMode === 'precontrolled') {
    $availableMACs = getCompanyMACs($config['comp_group']);
}
?>

<div class="device-overview">
    <!-- Device MAC Display -->
    <div class="mac-display">
        <div style="font-size: 12px; opacity: 0.7; margin-bottom: 4px;">
            Device MAC Address
        </div>
        
        <?php if ($config['features']['mac_modes']['precontrolled'] && $config['features']['mac_modes']['entered']): ?>
            <!-- Both modes - show dropdown with "Add New" option -->
            <select id="mac-selector" class="form-input" style="font-family: 'Courier New', monospace; 
                    background-color: rgba(255,255,255,0.1); color: var(--accent-color); 
                    font-size: 16px; text-align: center; cursor: pointer;"
                    onchange="handleMACChange(this.value)">
                <option value="">-- Select Device --</option>
                <?php 
                // Show pre-controlled MACs
                if (!empty($availableMACs)) {
                    echo '<optgroup label="Company Devices">';
                    foreach ($availableMACs as $mac) {
                        $selected = ($mac['MAC'] === $currentMAC) ? 'selected' : '';
                        echo '<option value="' . htmlspecialchars($mac['MAC']) . '" ' . $selected . '>';
                        echo htmlspecialchars($mac['SN'] . ' - ' . $mac['MAC']);
                        echo '</option>';
                    }
                    echo '</optgroup>';
                }
                ?>
                <option value="ADD_NEW">➕ Add Custom MAC...</option>
            </select>
            
            <!-- Hidden input for new MAC -->
            <div id="new-mac-input" style="display:none; margin-top:10px;">
                <input type="text" id="custom-mac" class="form-input" 
                       placeholder="AA:BB:CC:DD:EE:FF" 
                       style="font-family: 'Courier New', monospace;">
                <button onclick="addCustomMAC()" class="btn btn-small" style="margin-top:5px;">Add</button>
            </div>
            
        <?php elseif ($macMode === 'precontrolled' && !empty($availableMACs)): ?>
            <!-- Pre-controlled only - simple dropdown -->
            <select id="mac-selector" class="form-input" style="font-family: 'Courier New', monospace; 
                    background-color: rgba(255,255,255,0.1); color: var(--accent-color); 
                    font-size: 16px; text-align: center; cursor: pointer;"
                    onchange="switchMAC(this.value)">
                <?php foreach ($availableMACs as $mac): ?>
                    <option value="<?php echo htmlspecialchars($mac['MAC']); ?>" 
                            <?php echo $mac['MAC'] === $currentMAC ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($mac['SN']); ?> - <?php echo htmlspecialchars($mac['MAC']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        <?php else: ?>
            <!-- No MAC set -->
            <div style="color: var(--warning-color);">No device configured</div>
        <?php endif; ?>
    </div>
    
    <!-- NOW LIVE Section -->
    <div style="margin-bottom: 30px;">
        <h3 style="font-size: 16px; margin-bottom: 16px;">NOW LIVE</h3>
        
        <?php if (empty($selections)): ?>
        <div style="text-align: center; padding: 40px 20px; background: rgba(255,255,255,0.05); 
                    border-radius: 12px; color: rgba(255,255,255,0.5);">
            No selections active
        </div>
        <?php else: ?>
        <div class="selection-preview" style="background: rgba(255,255,255,0.05); 
                                            border-radius: 12px; padding: 20px;">
            <div style="display: flex; gap: 10px; overflow-x: auto; padding-bottom: 10px;">
                <?php foreach ($selections as $selectionId): ?>
                    <?php if (isset($contentMap[$selectionId])): ?>
                        <?php $content = $contentMap[$selectionId]; ?>
                        <div style="flex: 0 0 auto; text-align: center;">
                            <?php if ($content['content_type'] === 'image' && $content['content_path']): ?>
                                <img src="<?php echo $config['paths']['uploads'] . basename($content['content_path']); ?>" 
                                     style="height: 32px; width: auto; display: block;">
                            <?php elseif ($content['content_type'] === 'text'): ?>
                                <div style="background: #000; color: <?php echo $content['content_data']['color'] ?? '#FFF'; ?>; 
                                          padding: 4px 8px; font-size: 10px; height: 32px; 
                                          display: flex; align-items: center; white-space: nowrap;">
                                    <?php echo htmlspecialchars(substr($content['content_data']['text'] ?? '', 0, 30)); ?>
                                </div>
                            <?php endif; ?>
                            <small style="font-size: 10px; opacity: 0.7; margin-top: 4px; display: block;">
                                <?php echo htmlspecialchars($content['content_name']); ?>
                            </small>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Action Buttons -->
    <div style="display: flex; gap: 12px; margin-bottom: 20px;">
        <a href="?page=content-library" class="btn" style="flex: 1;">
            Edit Selections
        </a>
    </div>
    
    <!-- Save Button -->
    <button type="button" id="save-selections-btn" class="btn btn-secondary" onclick="saveSelections()">
        Save Selections
    </button>
</div>

<script>
// Set current selections in app state
AppState.selections = <?php echo json_encode(array_values($selections)); ?>;
AppState.currentMAC = '<?php echo $currentMAC; ?>';

function handleMACChange(value) {
    if (value === 'ADD_NEW') {
        document.getElementById('new-mac-input').style.display = 'block';
        document.getElementById('mac-selector').value = '<?php echo $currentMAC; ?>';
    } else if (value) {
        switchMAC(value);
    }
}

function addCustomMAC() {
    const mac = document.getElementById('custom-mac').value.trim().toUpperCase();
    if (mac.match(/^([0-9A-F]{2}[:-]){5}([0-9A-F]{2})$/i)) {
        switchMAC(mac, 'entered');
        document.getElementById('new-mac-input').style.display = 'none';
    } else {
        showStatus('Invalid MAC format', 'error');
    }
}

// Function to switch MAC address
async function switchMAC(newMAC) {
    if (newMAC === AppState.currentMAC) return;
    
    try {
        // Show loading state
        const selector = document.getElementById('mac-selector');
        selector.disabled = true;
        
        // Call device API to update MAC
        const response = await fetch('api/device.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
                mac: newMAC, 
                mode: 'precontrolled' 
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Reload page to show new device data
            window.location.reload();
        } else {
            // Revert selection on error
            selector.value = AppState.currentMAC;
            selector.disabled = false;
            showStatus(result.error || 'Failed to switch device', 'error');
        }
    } catch (error) {
        // Revert selection on error
        document.getElementById('mac-selector').value = AppState.currentMAC;
        document.getElementById('mac-selector').disabled = false;
        showStatus('Connection error', 'error');
    }
}
</script>