<?php
require_once '../../config.php';
session_start();
header('Content-Type: application/json; charset=UTF-8');

// 1) validate inputs
if (
  empty($_GET['mac'])
  || !preg_match('/^([0-9A-Fa-f]{2}:){5}[0-9A-Fa-f]{2}$/', $_GET['mac'])
  || !isset($_GET['session'])
  || !ctype_digit($_GET['session'])
) {
  http_response_code(400);
  exit;
}
$mac     = $_GET['mac'];
$session = (int)$_GET['session'];

// 2) connect
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) { http_response_code(500); exit; }

// 3) compute Eastern‐time
$dt = new DateTime('now', new DateTimeZone('UTC'));
$dt->setTimezone(new DateTimeZone('America/New_York'));
$fmt = $dt->format('n/j/Y g:i A');

// 4) Check if IP has changed and needs location update
$currentIpPacked = inet_pton($_SERVER['REMOTE_ADDR']);

$stmt = mysqli_prepare($con, "
  SELECT ip_address, region_name
  FROM glance_led_v2_tracking
  WHERE mac_address = ?
    AND session_count = ?
");
mysqli_stmt_bind_param($stmt, 'si', $mac, $session);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $storedIpPacked, $storedRegion);
mysqli_stmt_fetch($stmt);
mysqli_stmt_close($stmt);

// Check if IP changed or location is missing
$ipChanged = ($storedIpPacked !== $currentIpPacked);
$needsLocation = empty($storedRegion);

if ($ipChanged || $needsLocation) {
    // Lookup new location
    $locationData = get_ip_location($_SERVER['REMOTE_ADDR']);

    if ($locationData) {
        // Update IP and location
        $stmt = mysqli_prepare($con, "
          UPDATE glance_led_v2_tracking
          SET heartbeat = heartbeat + 1,
              last_ping_time = ?,
              ip_address = ?,
              region_name = ?,
              city = ?,
              isp = ?,
              zip = ?,
              lat = ?,
              lon = ?,
              timezone = ?
          WHERE mac_address = ?
            AND session_count = ?
        ");

        $region = $locationData['regionName'] ?? null;
        $city = $locationData['city'] ?? null;
        $isp = $locationData['isp'] ?? null;
        $zip = $locationData['zip'] ?? null;
        $lat = $locationData['lat'] ?? null;
        $lon = $locationData['lon'] ?? null;
        $timezone = $locationData['timezone'] ?? null;

        // Bind params: s s s s s s d d s s i
        mysqli_stmt_bind_param($stmt, 'ssssssddssi',
            $fmt, $currentIpPacked, $region, $city, $isp, $zip, $lat, $lon, $timezone, $mac, $session);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    } else {
        // Just update heartbeat and IP (couldn't get location)
        $stmt = mysqli_prepare($con, "
          UPDATE glance_led_v2_tracking
          SET heartbeat = heartbeat + 1,
              last_ping_time = ?,
              ip_address = ?
          WHERE mac_address = ?
            AND session_count = ?
        ");
        mysqli_stmt_bind_param($stmt, 'sssi', $fmt, $currentIpPacked, $mac, $session);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
} else {
    // 5) Normal update: just heartbeat + last_ping_time
    $stmt = mysqli_prepare($con, "
      UPDATE glance_led_v2_tracking
         SET heartbeat       = heartbeat + 1,
             last_ping_time  = ?
       WHERE mac_address    = ?
         AND session_count  = ?
    ");
    mysqli_stmt_bind_param($stmt, 'ssi', $fmt, $mac, $session);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
}

// 6) fetch new heartbeat & original session_starttime
$stmt = mysqli_prepare($con, "
  SELECT heartbeat, session_starttime
    FROM glance_led_v2_tracking
   WHERE mac_address   = ?
     AND session_count = ?
");
mysqli_stmt_bind_param($stmt, 'si', $mac, $session);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $hb, $sessStart);
mysqli_stmt_fetch($stmt);
mysqli_stmt_close($stmt);
mysqli_close($con);

// 7) Return response (ESP32 compatible - same format as before)
echo json_encode([
    'heartbeat' => (int)$hb,
    'session_start' => $sessStart
], JSON_NUMERIC_CHECK);

/**
 * Get IP geolocation data from ip-api.com
 */
function get_ip_location($ip) {
    // Skip private/local IPs
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return null;
    }

    $url = "http://ip-api.com/json/$ip?fields=status,message,country,countryCode,region,regionName,city,zip,lat,lon,timezone,isp";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code === 200 && $response) {
        $data = json_decode($response, true);
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            return $data;
        }
    }

    return null;
}
?>


