<?php
require_once '../../config.php';
session_start();
header('Content-Type: application/json; charset=UTF-8');

// 1) ignore your factory IP (optional)
// if ($_SERVER['REMOTE_ADDR'] === '50.185.150.242') {
//     http_response_code(204);
//     exit;
// }

// 2) validate MAC
if (
    empty($_GET['mac']) ||
    !preg_match('/^([0-9A-Fa-f]{2}:){5}[0-9A-Fa-f]{2}$/', $_GET['mac'])
) {
    http_response_code(400);
    exit;
}
$mac = $_GET['mac'];

// 3) compute next session_count
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    http_response_code(500);
    exit;
}
$stmt = mysqli_prepare($con, "
   SELECT COALESCE(MAX(session_count),0) + 1
     FROM glance_led_v2_tracking
    WHERE mac_address = ?
");
mysqli_stmt_bind_param($stmt, 's', $mac);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $nextSession);
mysqli_stmt_fetch($stmt);
mysqli_stmt_close($stmt);

// 4) pack the caller's IP
$ipPacked = inet_pton($_SERVER['REMOTE_ADDR']);
if ($ipPacked === false) {
    http_response_code(400);
    exit;
}

// 5) format Eastern time
$dt = new DateTime('now', new DateTimeZone('UTC'));
$dt->setTimezone(new DateTimeZone('America/New_York'));
$fmt = $dt->format('n/j/Y g:i A');

// 6) Check if this IP already has location data in database
$region_val = null;
$city_val = null;
$isp_val = null;
$zip_val = null;
$lat_val = null;
$lon_val = null;
$timezone_val = null;

$stmt = mysqli_prepare($con, "
    SELECT region_name, city, isp, zip, lat, lon, timezone
    FROM glance_led_v2_tracking
    WHERE ip_address = ?
    AND region_name IS NOT NULL
    LIMIT 1
");
mysqli_stmt_bind_param($stmt, 's', $ipPacked);
mysqli_stmt_execute($stmt);
mysqli_stmt_bind_result($stmt, $region_val, $city_val, $isp_val, $zip_val, $lat_val, $lon_val, $timezone_val);
$hasLocation = mysqli_stmt_fetch($stmt);
mysqli_stmt_close($stmt);

// If no cached location, look it up from API
if (!$hasLocation) {
    $locationData = get_ip_location($_SERVER['REMOTE_ADDR']);
    if ($locationData) {
        $region_val = $locationData['regionName'] ?? null;
        $city_val = $locationData['city'] ?? null;
        $isp_val = $locationData['isp'] ?? null;
        $zip_val = $locationData['zip'] ?? null;
        $lat_val = $locationData['lat'] ?? null;
        $lon_val = $locationData['lon'] ?? null;
        $timezone_val = $locationData['timezone'] ?? null;
    }
}

// 7) Insert the row
// Column order: mac_address, ip_address, region_name, city, isp, zip, lat, lon, timezone, session_starttime, session_count, heartbeat, last_ping_time
$ins = mysqli_prepare($con, "
  INSERT INTO glance_led_v2_tracking
    (mac_address, ip_address, region_name, city, isp, zip, lat, lon, timezone,
     session_starttime, session_count, heartbeat, last_ping_time)
  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, ?)
");

// Bind parameters: s s s s s s d d s s i s
mysqli_stmt_bind_param($ins,
  'ssssssddssis',
  $mac,
  $ipPacked,
  $region_val,
  $city_val,
  $isp_val,
  $zip_val,
  $lat_val,
  $lon_val,
  $timezone_val,
  $fmt,
  $nextSession,
  $fmt
);
mysqli_stmt_execute($ins);
mysqli_stmt_close($ins);
mysqli_close($con);

// 8) emit only a numeric session_id (ESP32 compatible)
echo json_encode(
  ['session_id' => (int)$nextSession],
  JSON_NUMERIC_CHECK
);

/**
 * Get IP geolocation data from ip-api.com
 * Only called for NEW IPs that haven't been looked up before
 */
function get_ip_location($ip) {
    // Skip private/local IPs
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return null;
    }

    $url = "http://ip-api.com/json/$ip?fields=status,message,country,countryCode,region,regionName,city,zip,lat,lon,timezone,isp";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code === 200 && $response) {
        $data = json_decode($response, true);
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            return $data;
        }
    }

    return null;
}
?>
