<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once $_SERVER['DOCUMENT_ROOT'] . '/config.php';

set_time_limit(300);

// Connect to database
$con = mysqli_connect($host_gscroll, $username_gscroll, $password_gscroll, $database_gscroll);
if (!$con) {
    die("Connection failed: " . mysqli_connect_error());
}

// Function to convert binary IP to decimal notation
function binaryToIp($binary) {
    if (empty($binary) || strlen($binary) != 4) {
        return false; // IPv4 must be exactly 4 bytes
    }
    
    // Convert binary to IP string
    $ip = inet_ntop($binary);
    return $ip;
}

// Function to validate if IP is public (not reserved/private)
function isPublicIp($ip) {
    if (!$ip || !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        return false;
    }
    
    // Convert IP to long for range checking
    $long = ip2long($ip);
    
    // Check if it's in private/reserved ranges
    $privateRanges = [
        ['0.0.0.0', '0.255.255.255'],      // Reserved
        ['10.0.0.0', '10.255.255.255'],    // Private
        ['127.0.0.0', '127.255.255.255'],  // Loopback
        ['169.254.0.0', '169.254.255.255'], // Link-local
        ['172.16.0.0', '172.31.255.255'],  // Private
        ['192.168.0.0', '192.168.255.255'], // Private
    ];
    
    foreach ($privateRanges as $range) {
        $start = ip2long($range[0]);
        $end = ip2long($range[1]);
        if ($long >= $start && $long <= $end) {
            return false;
        }
    }
    
    return true;
}

// Get unique IP addresses that need geolocation
// IMPORTANT: Use BINARY mode when selecting
$query = "SELECT DISTINCT ip_address FROM glance_led_v2_tracking 
          WHERE (region_name IS NULL OR city IS NULL OR isp IS NULL) 
          AND ip_address IS NOT NULL 
          AND LENGTH(ip_address) = 4";

$result = mysqli_query($con, $query);

if (!$result) {
    die("Query failed: " . mysqli_error($con));
}

// Filter to only valid public IPs
$uniqueIps = [];
$skippedCount = 0;
$debugCount = 0;

echo "=== Processing IP addresses ===\n\n";

while ($row = mysqli_fetch_assoc($result)) {
    $binaryIp = $row['ip_address'];
    $decimalIp = binaryToIp($binaryIp);
    
    // Debug first 10
    if ($debugCount < 10) {
        echo "Binary: " . bin2hex($binaryIp) . " → Decimal: " . ($decimalIp ?: 'FAILED');
        if ($decimalIp) {
            echo " → Public: " . (isPublicIp($decimalIp) ? 'YES' : 'NO');
        }
        echo "\n";
        $debugCount++;
    }
    
    if ($decimalIp && isPublicIp($decimalIp)) {
        $uniqueIps[$binaryIp] = $decimalIp;
        
        // LIMIT TO 200 FOR TESTING
        if (count($uniqueIps) >= 200) {
            break;
        }
    } else {
        $skippedCount++;
    }
}

echo "\nFound " . count($uniqueIps) . " valid public IP addresses to lookup\n";
echo "Skipped $skippedCount invalid/private IPs\n\n";

if (count($uniqueIps) == 0) {
    echo "No valid IPs to process!\n";
    mysqli_close($con);
    exit;
}

// Process in batches of 100
$ipBatches = array_chunk($uniqueIps, 100, true);
$processedCount = 0;

foreach ($ipBatches as $batchIndex => $batch) {
    echo "Processing batch " . ($batchIndex + 1) . " of " . count($ipBatches) . "\n";
    
    // Prepare batch request
    $ipsToLookup = array_values($batch);
    
    echo "Sample IPs in this batch:\n";
    foreach (array_slice($ipsToLookup, 0, 5) as $ip) {
        echo "  $ip\n";
    }
    echo "\n";
    
    // Make batch API call
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "http://ip-api.com/batch");
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($ipsToLookup));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        echo "API request failed with code $httpCode\n";
        echo "Response: $response\n";
        continue;
    }
    
    $results = json_decode($response, true);
    
    if (!$results) {
        echo "Failed to decode API response\n";
        continue;
    }
    
    // Map results back to binary IPs and update database
    $binaryIps = array_keys($batch);
    foreach ($results as $index => $data) {
        $binaryIp = $binaryIps[$index];
        
        if ($data['status'] === 'success') {
            // Prepare data for update
            $region = mysqli_real_escape_string($con, $data['regionName'] ?? '');
            $city = mysqli_real_escape_string($con, $data['city'] ?? '');
            $isp = mysqli_real_escape_string($con, $data['isp'] ?? '');
            $zip = mysqli_real_escape_string($con, $data['zip'] ?? '');
            $lat = isset($data['lat']) ? (float)$data['lat'] : 'NULL';
            $lon = isset($data['lon']) ? (float)$data['lon'] : 'NULL';
            $timezone = mysqli_real_escape_string($con, $data['timezone'] ?? '');
            
            // Update all rows with this IP address (using binary comparison)
            $escapedBinary = mysqli_real_escape_string($con, $binaryIp);
            $updateQuery = "UPDATE glance_led_v2_tracking SET 
                region_name = '$region',
                city = '$city',
                isp = '$isp',
                zip = '$zip',
                lat = $lat,
                lon = $lon,
                timezone = '$timezone'
                WHERE ip_address = '$escapedBinary'";
            
            if (mysqli_query($con, $updateQuery)) {
                $affected = mysqli_affected_rows($con);
                $processedCount += $affected;
                echo "✓ Updated $affected rows for IP " . $data['query'] . " → $city, $region ($isp)\n";
            } else {
                echo "✗ Update failed: " . mysqli_error($con) . "\n";
            }
        } else {
            echo "✗ Failed to lookup IP " . $batch[$binaryIp] . ": " . ($data['message'] ?? 'unknown error') . "\n";
        }
    }
    
    echo "\n";
}

echo "✓ Completed! Updated $processedCount total rows\n";

mysqli_close($con);
?>