// Execute immediately on script load - don't wait for DOMContentLoaded
(function() {
    const style = document.createElement('style');
    style.textContent = `
        .right-panel {
            display: flex;
            flex-direction: column;
            height: 100%;
            overflow: hidden;
        }
        
        .panel-section.block-library-section {
            flex: 1;
            min-height: 100px;
            position: relative;
            overflow-y: auto;
        }
        
        .panel-section.properties-section {
            flex: 0 0 50%;  /* Start at 50% */
            min-height: 100px;
            overflow-y: auto;
        }
        
        .resize-handle {
            position: absolute;
            left: 0;
            right: 0;
            bottom: 0;
            height: 5px;
            background-color: var(--bg-light);
            cursor: ns-resize;
            z-index: 100;
        }
        
        .resize-handle:hover {
            background-color: var(--primary);
        }
    `;
    document.head.appendChild(style);

    function removePropertiesTitle() {
        // Initial removal
        const removeTitle = () => {
            const titles = document.querySelectorAll('h3.properties-title, .properties-title');
            titles.forEach(title => title.remove());
        };

        // Remove immediately
        removeTitle();

        // Set up observer to remove title whenever it's added
        const observer = new MutationObserver((mutations) => {
            mutations.forEach(() => {
                removeTitle();
            });
        });

        // Start observing the properties container
        const propertiesContainer = document.querySelector('.properties-container');
        if (propertiesContainer) {
            observer.observe(propertiesContainer, {
                childList: true,
                subtree: true
            });
        }
    }

    function setupResizableRightPanel() {
        const blockLibrary = document.querySelector('.panel-section.block-library-section');
        const properties = document.querySelector('.panel-section.properties-section');
        const rightPanel = document.querySelector('.right-panel');
        
        if (!blockLibrary || !properties || !rightPanel) {
            console.error('Missing elements:', {blockLibrary, properties, rightPanel});
            return;
        }

        // Remove the static properties title
        removePropertiesTitle();

        console.log('Initial sizes:', {
            rightPanel: rightPanel.offsetHeight,
            blockLibrary: blockLibrary.offsetHeight,
            properties: properties.offsetHeight
        });
        
        // Set initial 50/50 split
        const totalHeight = rightPanel.offsetHeight;
        const halfHeight = Math.floor(totalHeight / 2);
        blockLibrary.style.height = `${halfHeight}px`;
        properties.style.height = `${halfHeight}px`;

        // Create resize handle
        const resizeHandle = document.createElement('div');
        resizeHandle.className = 'resize-handle';
        blockLibrary.appendChild(resizeHandle);
        
        let isResizing = false;
        let startY = 0;
        let startBlockHeight = 0;
        let startPropsHeight = 0;
        
        resizeHandle.addEventListener('mousedown', (e) => {
            isResizing = true;
            startY = e.clientY;
            startBlockHeight = blockLibrary.offsetHeight;
            startPropsHeight = properties.offsetHeight;
            
            console.log('Started resize:', {
                startY,
                startBlockHeight,
                startPropsHeight
            });
            
            document.body.style.cursor = 'ns-resize';
            
            // Force remove any inline styles that might interfere
            blockLibrary.style.cssText = '';
            properties.style.cssText = '';
        });
        
        document.addEventListener('mousemove', (e) => {
            if (!isResizing) return;
            
            const totalHeight = rightPanel.offsetHeight;
            const deltaY = e.clientY - startY;
            
            // Calculate new heights
            let newBlockHeight = startBlockHeight + deltaY;
            let newPropsHeight = startPropsHeight - deltaY;
            
            console.log('Resizing:', {
                deltaY,
                newBlockHeight,
                newPropsHeight,
                totalHeight
            });
            
            // Enforce min/max constraints
            const minHeight = 100;
            if (newBlockHeight >= minHeight && newPropsHeight >= minHeight) {
                // Set flex-grow and flex-basis for block library
                blockLibrary.style.flex = 'none';
                blockLibrary.style.height = `${newBlockHeight}px`;
                
                // Set fixed height for properties
                properties.style.flex = 'none';
                properties.style.height = `${newPropsHeight}px`;
            }
        });
        
        document.addEventListener('mouseup', () => {
            if (isResizing) {
                console.log('Finished resize:', {
                    finalBlockHeight: blockLibrary.offsetHeight,
                    finalPropsHeight: properties.offsetHeight
                });
            }
            isResizing = false;
            document.body.style.cursor = '';
        });
    }

    // Initialize when DOM is ready
    document.addEventListener('DOMContentLoaded', setupResizableRightPanel);
})();

// Add this at the end of the file to ensure our divider stays visible
document.addEventListener('DOMContentLoaded', function() {
    // Final check to ensure divider visibility
    const divider = document.querySelector('.simple-divider-handle');
    if (divider) {
        divider.style.zIndex = '9999';
        divider.style.position = 'relative';
    }
}); 